/* main.c - Hello World + PLL example for S32R274 */
/* Description:  Drive CLKOUT0 to external pin with PLL */
/* Rev 1.0 Aug 10 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2018 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main.c
* Owner             David Chung
* Version           1.0
* Date              Aug-10-2018
* Classification    General Business Information
* Brief             S32R274 configure PLL and drives it to CLKOUT0
********************************************************************************
* Detailed Description:
* S32R274's default clock source is 16 MHz IRCOSC.  This code example
* configures the PLL0 to 240 MHz. The system clock selector selects
* PLL0_PHI_CLK as its source. The clock domain SYS_CLK divides that by 3,
* such that SYS_CLK = 60 MHz. The z4 core drives PLL0_PHI_CLK out divided by 20
* (240 MHz/20 = 12 MHz). A GPIO also toggles to indicate that the program is
* working. Connect to a user LEd on the MPC57xxMB to observe it flash.
* The frequency of the toggle is based on an internal counter.
*
* Refer to "Z4 Waveform.png" for expected CLKOUT output
*
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB + MPC57XXMB
* MCU:             S32R274
* Terminal:        None
* Fsys:            240 MHz PLL
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Connect pin P9.7 (CLK_OUT0) to an oscilloscope to check
* 					CLK_OUT0 frequency. Connect P8.1 (PA0) to LED P7.1 (D2)
*
* 					Refer to "Hardware Setup.jpg"
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-10-2018  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */

#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

#define COUNTER_MAX 500000

extern void xcptn_xmpl(void);

void CLK_Init_Z4(void);

void CLKOUT0_Init_Z4(void);

void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

void CLK_Init_Z4(){
	/* Configure PLL0 to 240 MHz */
	CGM.AC3_SC.R = 0x01000000; //Select XOSC (40 MHz) as source of PLL0

	/* f_PLL0_PHI0 = fin * (MFD/(PREDIV*RFDPHI))
	 * 				= 40 * (12/(2*1))
	 * 				= 240
	 *
	 * 	f_PLL0_PHI1 = fin * (MFD/(PREDIV*RFDPHI1))
	 * 				= 40 * (12/(2*6))
	 * 				= 40
	 */
	PLLDIG.PLL0DV.R = 0x3001200C;

	/* Turn on clock sources */
	MC_ME.DRUN_MC.R = 0x00130072; //Turn on IRC, XOSC, and PLL0. Set system clock to PLL0_PHI_CLK (240 MHz)

	/* Perform mode switch for changes to take effect*/
	MC_ME.MCTL.R = 0x30005AF0;
	MC_ME.MCTL.R = 0x3000A50F;
	while(MC_ME.GS.B.S_MTRANS); //Wait for transition to complete

	/* Configure SYS_CLK */
	CGM.SC_DC0.R = 0x80030000; //Enable and divide by 4 (60 MHz)
}

void CLKOUT0_Init_Z4(){
	/* S32R27 boots from the 16 MHz IRCOSC.
	 * MCB controls CLKOUT source selects.
	 */
	MCB.CLKOUT_SEL.B.CLKOUT0 = 0b0010; //Select PLL0 for CLKOUT0
	CGM.AC14_DC0.R = 0x80130000; //Divide by (19+1) = 20 for output of 12 MHz

	/* CLKOUT0 maps to PB6 (MSCR22) */
	SIUL2.MSCR[22].B.SSS = 1; //Configure pin for CLKOUT0
	SIUL2.MSCR[22].B.OBE = 1; //Enable output buffer

	/* Configure GPIO PA0 (P8.1) */
	SIUL2.MSCR[0].B.OBE = 1;
	SIUL2.GPDO[0].B.PDO_4n = 1;
}

int main(void)
{
	int counter = 0;

	xcptn_xmpl ();              /* Configure and Enable Interrupts */

	CLK_Init_Z4();				/* Configure PLL0 to 240 MHz and SYS_CLK = PLL0_PHI0 */

	/* Configure CLKOUT and peripherals */
	CLKOUT0_Init_Z4();			/* Configure CLKOUT0 pin */

	/* Loop forever */
	for(;;) {	   
	   	counter++;

	   	/* Wait 500000 counts of internal counter
	   	 * before toggling GPIO.
	   	 */
	   	if(counter >= COUNTER_MAX){
	   		SIUL2.GPDO[0].B.PDO_4n ^= 1; //Toggle PA0
	   		counter = 0; //Reset counter
	   	}
	}
}


